#!/usr/bin/env python
# -*- coding: utf-8 -*-
import comm

from update_logger import Logger
import ostech_connection
import flash_connection
from refresh_icon import refresh_bitmap
from led import LED
import wx


def EVT_CONNECTION_STATE_EVENT(win, func):
    win.Connect(-1, -1, ostech_connection.wxCONNECTION_STATE_EVENT, func)


def EVT_PROGRAMMING_MODE_EVENT(win, func):
    win.Connect(-1, -1, ostech_connection.wxPROGRAMMING_MODE_EVENT, func)


def EVT_FLASH_CONNECTION_STATE_EVENT(win, func):
    win.Connect(-1, -1, flash_connection.wxFLASH_CONNECTION_STATE_EVENT, func)


def EVT_FLASH_ERASE_BLOCKS_EVENT(win, func):
    win.Connect(-1, -1, flash_connection.wxFLASH_ERASE_BLOCKS_EVENT, func)


def EVT_FLASH_WRITE_BLOCKS_EVENT(win, func):
    win.Connect(-1, -1, flash_connection.wxFLASH_WRITE_BLOCKS_EVENT, func)


def EVT_FLASH_REPAIR_BLOCKS_EVENT(win, func):
    win.Connect(-1, -1, flash_connection.wxFLASH_REPAIR_BLOCKS_EVENT, func)


def EVT_FLASH_VERIFY_BLOCKS_EVENT(win, func):
    win.Connect(-1, -1, flash_connection.wxFLASH_VERIFY_BLOCKS_EVENT, func)


def EVT_FLASH_VERIFY_FINISHED_EVENT(win, func):
    win.Connect(-1, -1, flash_connection.wxFLASH_VERIFY_FINISHED_EVENT, func)


def EVT_FLASH_REPAIR_FINISHED_EVENT(win, func):
    win.Connect(-1, -1, flash_connection.wxFLASH_REPAIR_FINISHED_EVENT, func)


class PortSelect(wx.Choice):
    def __init__(self, parent, choices=[]):
        wx.Choice.__init__(self, parent, -1, choices=choices)
        self.update_ports()

    def update_ports(self):
        ports = comm.enumerate_serial_ports()

        oldIndex = self.GetCurrentSelection()
        if oldIndex != wx.NOT_FOUND:
            oldChoice = self.GetString(oldIndex)

        self.Clear()
        self.AppendItems(ports)

        if oldIndex != wx.NOT_FOUND:
            newIndex = self.FindString(oldChoice)
            if newIndex != wx.NOT_FOUND:
                self.SetSelection(newIndex)

        if self.GetCurrentSelection() == wx.NOT_FOUND and \
                self.GetCount() > 0:
            self.SetSelection(0)


class LogCtrl(wx.TextCtrl):
    def __init__(
            self, parent, style=wx.TE_MULTILINE | wx.TE_READONLY, logger=None):
        wx.TextCtrl.__init__(self, parent, -1, style=style)
        if logger is not None:
            logger.register(self)

    def log(self, text):
        self.AppendText(text + "\n")


class Frame(wx.Frame):
    """Main frame of this application."""

    def __init__(self, title, size):
        wx.Frame.__init__(self, None, -1, title, size=size)
        self.SetMinSize(wx.Size(600, 400))
        self.logger = Logger()
        self.connection = None
        self.flashConnection = None
        self.verificationErrors = 0
        self.disconnectAfterConnect = False

        introtext = "Please disconnect the laser and the TEC from the " + \
            "device that you want to update. Then turn the device on and " + \
            "connect it to a serial port of this computer."

        self.panel = wx.Panel(self, -1)
        self.portCaption = wx.StaticText(
            self.panel, label="Serial port:")
        self.refreshButton = wx.BitmapButton(
            self.panel, -1, bitmap=refresh_bitmap.GetBitmap())
        self.portSelect = PortSelect(self.panel)
        self.connectButton = wx.Button(self.panel, -1, label="Connect")

        self.connectedLed = LED(self.panel)
        self.driverType = wx.StaticText(self.panel, -1)
        self.driverSn = wx.StaticText(self.panel, -1)
        self.driverSoftware = wx.StaticText(self.panel, -1)
        self.set_device_infos(None, None, None)

        self.updateButton = wx.Button(self.panel, -1, label="Update Firmware")
        self.updateButton.Disable()

        self.textDoing = wx.StaticText(self.panel, label=u"...",
                                       style=wx.ALIGN_CENTRE_HORIZONTAL)
        self.progress = wx.Gauge(self.panel, -1)

        self.log = LogCtrl(self.panel, logger=self.logger)
        self.logger.log(introtext)

        border = 4
        self.connectBox = wx.BoxSizer(wx.HORIZONTAL)
        self.connectBox.Add(
            self.portCaption, 1, wx.ALL | wx.ALIGN_CENTER_VERTICAL,
            border=border)
        self.connectBox.Add(
            self.refreshButton, 0, wx.ALL | wx.ALIGN_CENTER_VERTICAL,
            border=border)
        self.connectBox.Add(
            self.portSelect, 2, wx.ALL | wx.ALIGN_CENTER_VERTICAL,
            border=border)
        self.connectBox.Add(
            self.connectButton, 2,
            wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=border)
        self.connectBox.Add(
            self.connectedLed, 0,
            wx.LEFT | wx.RIGHT | wx.ALIGN_CENTER_VERTICAL, border=border)

        flags = wx.ALL | wx.ALIGN_CENTER_VERTICAL
        self.driverBox = wx.BoxSizer(wx.HORIZONTAL)
        self.driverBox.Add(self.driverType, 1, flags, border=border)
        self.driverBox.Add(self.driverSn, 1, flags, border=border)
        self.driverBox.Add(self.driverSoftware, 1, flags, border=border)
        self.driverBox.Add(self.updateButton, 1, flags, border=border)

        flags = wx.ALL | wx.EXPAND
        self.vbox = wx.BoxSizer(wx.VERTICAL)
        self.vbox.Add(self.connectBox, 0, flags, border=border)
        self.vbox.Add(wx.StaticLine(self.panel), 0, flags, border=border)
        self.vbox.Add(self.driverBox, 0, flags, border=border)
        self.vbox.Add(wx.StaticLine(self.panel), 0, flags, border=border)
        self.vbox.Add(self.textDoing, 0, wx.ALIGN_CENTRE_HORIZONTAL | wx.ALL,
                      border=border)
        self.vbox.Add(self.progress, 0, flags, border=border)
        self.vbox.Add(self.log, 1, flags, border=border)

        self.panel.SetSizer(self.vbox)

        self.refreshButton.Bind(wx.EVT_BUTTON, self.update_ports)
        self.portSelect.Bind(wx.EVT_CHOICE, self.choose_port)
        self.connectButton.Bind(wx.EVT_BUTTON, self.connect)
        self.updateButton.Bind(wx.EVT_BUTTON, self.update_firmware)
        self.Bind(wx.EVT_CLOSE, self.on_window_close)

        self.panelSizer = wx.BoxSizer(wx.VERTICAL)
        self.panelSizer.Add(self.panel, 1, wx.ALL | wx.EXPAND, border=0)
        self.SetSizer(self.panelSizer)

    def on_window_close(self, evt):
        if self.flashConnection is None:
            evt.Skip()
            return
        if not self.flashConnection.connected:
            evt.Skip()
            return
        message = "The firmware of the device is beeing updated. " \
            + "If this process is cancelled, there will probably be no " \
            + "proper firmware on the device. It can therefore not be " \
            + "reprogrammed and has to be send to the manufacturer.\n\n" \
            + "Do you really want to exit this program while updating the " \
            + "firmware?"
        if evt.CanVeto():
            answer = wx.MessageBox(message, "Exit while updating?",
                wx.ICON_QUESTION | wx.YES_NO)
            if answer != wx.YES:
                evt.Veto()
                return

        evt.Skip()

    def set_device_infos(self, deviceType, serialNumber, softwareVersion):
        labelDeviceType = "Device Type:"
        labelSerialNumber = "Serial Number:"
        labelSoftwareVersion = "Software Version:"

        text = labelDeviceType
        if deviceType is None:
            self.driverType.Disable()
        else:
            text += " " + str(deviceType)
            self.driverType.Enable()
            self.logger.log(text)
        self.driverType.SetLabel(text)

        text = labelSerialNumber
        if serialNumber is None:
            self.driverSn.Disable()
        else:
            text += " " + str(serialNumber)
            self.driverSn.Enable()
            self.logger.log(text)
        self.driverSn.SetLabel(text)

        text = labelSoftwareVersion
        if softwareVersion is None:
            self.driverSoftware.Disable()
        else:
            text += " " + str(softwareVersion)
            self.driverSoftware.Enable()
            self.logger.log(text)
        self.driverSoftware.SetLabel(text)

    def update_ports(self, evt):
        if self.portSelect is None:
            return
        self.portSelect.update_ports()
        self.choose_port(evt)

    def update_firmware(self, evt):
        self.logger.log(u"Switching to firmware flashing mode...")
        self.updateButton.Disable()
        self.connection.set_programming_mode()

    def choose_port(self, evt):
        self.updateButton.Disable()
        if self.portSelect.GetCurrentSelection() == wx.NOT_FOUND:
            self.connectButton.Disable()
        else:
            self.connectButton.Enable()

    def connection_lost(self, message):
        self.update_ports(None)
        self.logger.log(message)
        self.portCaption.Enable()
        self.refreshButton.Enable()
        self.connectButton.Enable()
        self.portSelect.Enable()
        self.connectedLed.SetState(False)
        self.updateButton.Disable()
        self.set_device_infos(None, None, None)

    def connection_established(self, message):
        self.logger.log(message)
        self.portCaption.Disable()
        self.refreshButton.Disable()
        self.connectButton.Disable()
        self.portSelect.Disable()
        self.connectedLed.SetState(True)
        self.updateButton.Enable()
        deviceType = self.connection.get_driver().deviceType
        serialNumber = self.connection.get_driver().serialNumber
        softwareVersion = self.connection.get_driver().softwareVersion
        buildId = self.connection.get_driver().buildId
        buildTime = self.connection.get_driver().buildTime
        self.set_device_infos(deviceType, serialNumber, softwareVersion)
        self.logger.log("Firmware ID: %s" % buildId)
        self.logger.log("Firmware Build Time: %s" % buildTime)
        if self.disconnectAfterConnect:
            self.disconnectAfterConnect = False
            event = ostech_connection.ConnectionStateEvent(
                False, "Connection closed.")
            self.connection_state(event)
        if deviceType != 708 or softwareVersion > 2030:
            message = "This firmware update is only intended for device " \
                + "type 708 with software version 2030 or less.\n" \
                + "Closing connection..."
            self.logger.log(message)
            event = ostech_connection.ConnectionStateEvent(
                False, "Connection closed.")
            self.connection_state(event)

    def connection_state(self, evt):
        if evt.connected:
            self.connection_established(evt.message)
        else:
            self.connection.disconnect()
            self.connection_lost(evt.message)

    def connect(self, evt):
        self.connectButton.Disable()
        self.connectButton.Navigate()
        self.connectButton.UpdateWindowUI()
        i = self.portSelect.GetCurrentSelection()
        if i == wx.NOT_FOUND:
            return
        port = self.portSelect.GetString(i)
        self.logger.log("Connecting to port " + str(port) + u"...")
        EVT_CONNECTION_STATE_EVENT(self, self.connection_state)
        EVT_PROGRAMMING_MODE_EVENT(self, self.programming_mode_entered)
        self.connection = ostech_connection.Connection(self, port)

    def repair_update(self, evt):
        self.progress.SetRange(evt.nTotal)
        self.progress.SetValue(evt.nWritten)
        self.textDoing.SetLabel(evt.message)
        self.vbox.Layout()

    def start_verify(self):
        EVT_FLASH_VERIFY_BLOCKS_EVENT(self, self.verify_update)
        EVT_FLASH_VERIFY_FINISHED_EVENT(self, self.verify_finished)
        self.logger.log("Verifying firmware...")
        self.flashConnection.verify()

    def repair_finished(self, evt):
        self.start_verify()

    def start_repair(self, pages):
        EVT_FLASH_REPAIR_BLOCKS_EVENT(self, self.repair_update)
        EVT_FLASH_REPAIR_FINISHED_EVENT(self, self.repair_finished)
        self.logger.log(u"Trying to repair errors...")
        self.flashConnection.rewrite(pages)

    def verify_update(self, evt):
        self.progress.SetRange(evt.nTotal)
        self.progress.SetValue(evt.nVerified)
        self.textDoing.SetLabel(evt.message)
        self.vbox.Layout()

    def verify_finished(self, evt):
        self.logger.log(evt.message)
        if evt.ok:
            self.logger.log(
                "The firmware was updated successfully. "
                + "Please turn the device off and disconnect it from this "
                + "computer.""")
            self.flashConnection.disconnect()
            self.disconnectAfterConnect = True
            wx.MilliSleep(1000)
            self.connect(None)
        else:
            self.logger.log(
                "There were errors while updating the firmware.")
            self.verificationErrors += 1
            if self.verificationErrors < 5:
                if type(evt.rewrite) == list:
                    self.start_repair(evt.rewrite)
                elif evt.rewrite == True:
                    self.logger.log(u"Repeating...")
                    self.start_erase()
            else:
                message = "The firmware of your device still has errors. " \
                    + "Please contact the manufacturer of your device."
                self.logger.log(message)
                self.flashConnection.disconnect()


    def write_update(self, evt):
        if evt.finished:
            self.logger.log("Writing firmware into flash finished.")
            self.start_verify()
        self.progress.SetRange(evt.nTotal)
        self.progress.SetValue(evt.nWritten)
        self.textDoing.SetLabel(evt.message)
        self.vbox.Layout()

    def erase_update(self, evt):
        if evt.finished:
            self.logger.log("Erasing firmware flash finished.")
            EVT_FLASH_WRITE_BLOCKS_EVENT(self, self.write_update)
            self.logger.log("Writing firmware into flash...")
            self.flashConnection.write()
        self.progress.SetRange(evt.nTotal)
        self.progress.SetValue(evt.nErased)
        self.textDoing.SetLabel(evt.message)
        self.vbox.Layout()

    def start_erase(self):
        self.logger.log(u"Erasing firmware flash...")
        EVT_FLASH_ERASE_BLOCKS_EVENT(self, self.erase_update)
        self.flashConnection.erase_blocks(120)

    def flash_connection_state(self, evt):
        if evt.connected:
            self.logger.log(evt.message)
            message = "The process of updating your device can take several " \
                + "minutes. Please don't cancel this process. If you do, " \
                + "there will probably be no proper firmware in your device " \
                + " and you can therefore not flash the firmware again and " \
                + "have to contact the manufacturer of the device."
            self.logger.log(message)
            self.start_erase()
        else:
            self.connection_lost(evt.message)

    def programming_mode_entered(self, evt):
        self.logger.log(evt.message)
        self.connection.disconnect()
        self.logger.log(u"Connecting to boot-loader...")
        EVT_FLASH_CONNECTION_STATE_EVENT(self, self.flash_connection_state)
        self.flashConnection = flash_connection.FlashConnection(
            self, self.connection.port)


class App(wx.App):

    mainFrame = None

    def OnInit(self):
        """Initialisation. Create frames, bind events."""
        self.frame = Frame('Toptica Firmware Update', (800, 600))
        App.mainFrame = self.frame
        self.frame.Show()
        self.SetTopWindow(self.frame)
        return True


if __name__ == "__main__":
    TA = App()
    TA.MainLoop()
