#!/usr/bin/env python
# -*- coding: utf-8 -*-
from driver import Driver
import threading
import Queue
from serial import SerialException
from comm import OsTechError
import wx


wxCONNECTION_STATE_EVENT = wx.NewEventType()
wxPROGRAMMING_MODE_EVENT = wx.NewEventType()


class DriverThreadCommands(object):
    def __call__(self):
        pass


class ConnectionStateEvent(wx.PyEvent):
    """Event informing the GUI about a new connection state"""
    def __init__(self, connected, message=None):
        wx.PyEvent.__init__(self)
        self.SetEventType(wxCONNECTION_STATE_EVENT)
        self.connected = connected
        self.message = message

    def Clone(self):
        self.__class__(self.GetId())


class ProgrammingModeEvent(wx.PyEvent):
    """Event informing the GUI about the programming mode"""
    def __init__(self, message=None):
        wx.PyEvent.__init__(self)
        self.SetEventType(wxPROGRAMMING_MODE_EVENT)
        self.message = message

    def Clone(self):
        self.__class__(self.GetId())


class DriverThreadConnectCommand(DriverThreadCommands):
    def __init__(self):
        DriverThreadCommands.__init__(self)

    def __call__(self, driverThread):
        port = driverThread.connection.port
        driver = Driver(port)
        if driver.is_connected():
            driver.readConfig()
            driver.readBuildId()
        if driverThread is not None:
            driverThread.driver = driver
        message = None
        connected = driver.is_connected()
        if connected:
            message = "Connection established to device on port " + port + "."
        else:
            message = "Could not connect to device on port " + port + "."
        event = ConnectionStateEvent(connected, message)
        driverThread.publish_event(event)


class DriverThreadCheckConnectionCommand(DriverThreadCommands):
    def __init__(self):
        DriverThreadCommands.__init__(self)

    def publish_event(self, driverThread):
            message = "Connection to device on port " \
                + driverThread.connection.port + " was lost."
            event = ConnectionStateEvent(False, message)
            driverThread.publish_event(event)
            driverThread.stop()

    def __call__(self, driverThread):
        driver = driverThread.driver
        if not driver.is_connected():
            self.publish_event(driverThread)
            return
        try:
            driver.reconnect()
        except SerialException, e:
            self.publish_event(driverThread)
            return
        if not driver.is_connected():
            self.publish_event(driverThread)


class DriverSetProgrammingModeCommand(DriverThreadCommands):
    def __init__(self):
        DriverThreadCommands.__init__(self)

    def __call__(self, driverThread):
        driver = driverThread.driver
        driver._comm.setProgrammingMode()
        message = "Switched to firmware flashing mode."
        event = ProgrammingModeEvent(message)
        driverThread.publish_event(event)
        driverThread.stop()


class DriverThread(threading.Thread):
    def __init__(self, connection):
        threading.Thread.__init__(self)
        self.daemon = True
        self._endEvent = threading.Event()
        self.connection = connection
        self.driver = None
        self._commandQ = Queue.Queue()
        self._idleCommand = DriverThreadCheckConnectionCommand()

    def publish_event(self, event):
        self.connection.publish_event(event)

    def enqueue_command(self, command):
        self._commandQ.put_nowait(command)

    def run(self):
        while not self._endEvent.isSet():
            if self.driver is not None:
                if not self.driver.is_connected():
                    continue
            try:
                cmd = self._commandQ.get(timeout=1)
            except Queue.Empty:
                if self.driver is not None:
                    cmd = self._idleCommand
            try:
                cmd(self)
            except (SerialException, OsTechError) as e:
                print e

    def stop(self):
        self._endEvent.set()

    def is_connected(self):
        if self.driver is None:
            return False
        return self.driver.is_connected()


class Connection(object):
    def __init__(self, parent, port):
        self.parent = parent
        self.port = port
        self.driverThread = DriverThread(self)
        self.driverThread.start()
        self._textCommandResponseEventReceiver = None
        self.connect()

    def publish_event(self, event):
        if self.parent is not None:
            wx.PostEvent(self.parent, event)

    def get_driver(self):
        return self.driverThread.driver

    def connect(self):
        cmd = DriverThreadConnectCommand()
        self.driverThread.enqueue_command(cmd)

    def disconnect(self):
        driver = self.driverThread.driver
        self.driverThread.stop()
        self.driverThread.join()
        if driver is not None:
            driver.disconnect()

    def is_connected(self):
        return self.driverThread.is_connected()

    def set_programming_mode(self):
        cmd = DriverSetProgrammingModeCommand()
        self.driverThread.enqueue_command(cmd)
