#!/usr/bin/env python
import serial
import comm
import cmds
from errors import OsTechError

class Driver(object):
    def __init__(self, port=0, idleThread=False):
        self.serialNumber = None
        self.softwareVersion = None
        self.deviceType = None
        self.buildId = None
        self.buildTime = None
        self.port = port
        self._isConnected = False
        self._comm = comm.Comm(port, idleThread)
        self._connectionErrors = 0
        self.modules = {}
        self.connect()

    def _tell_connection_status(self, ok):
        if ok:
            self._connectionErrors = 0
            self._isConnected = True
        else:
            if self._connectionErrors < 5:
                self._connectionErrors += 1
            if self._connectionErrors >= 5:
                self._isConnected = False

        self._isConnected = self._isConnected and self._comm.is_connected()

    def connect(self):
        if self._isConnected:
            return
        self._comm.connect()
        if not self._comm.is_connected():
            self._isConnected = False
            return
        self._isConnected = True
        try:
            self._isConnected = self._comm.detect_driver_once()
            if self._isConnected:
                self._comm.setBinaryMode()
        except serial.SerialTimeoutException:
            self._isConnected = False

    def disconnect(self):
        self._comm.disconnect()
        self._isConnected = False

    def reconnect(self):
        self._isConnected = self._comm.detect_driver_once()
        if self._isConnected:
            self._comm._flush_or_idle(True)
            try:
                self._comm.setBinaryMode()
            except (serial.SerialTimeoutException, OsTechError) as e:
                print e

    def is_connected(self):
        return self._isConnected

    def readBuildId(self):
        response = self._comm.readBuildId()
        self.buildId = response[0]
        self.buildTime = response[1]
        return response

    def sendCmd(self, command, newValue=None, retry=True):
        result = False
        if not self.is_connected():
            return result
        cmd = None
        if isinstance(command, cmds.Command):
            cmd = command
        elif isinstance(command, str):
            cmd = cmds.all[command]
        else: raise TypeError
        try:
            result = self._comm.sendCmd(cmd, newValue, retry=retry)
            self._tell_connection_status(ok=True)
        except (serial.SerialException, OsTechError), e:
            if isinstance(e, serial.SerialTimeoutException):
                self._tell_connection_status(ok=False)
                print command, self._connectionErrors
            raise
        return result

    def readConfig(self):
        """Read important configuration data from driver and determine present
        modules.
        """
        if self.is_connected():
            m = self._comm.sendCmd(cmds.general["GVN"])
            self.serialNumber = m.value
            m = self._comm.sendCmd(cmds.general["GVS"])
            self.softwareVersion = m.value
            m = self._comm.sendCmd(cmds.general["GVT"])
            self.deviceType = m.value


if __name__ == "__main__":
    print "OsTech\n" + 7 * "-"
    dr = Driver()
    dr.readConfig()

    print "Serial Number:", dr.serialNumber
    print "Software Version:", dr.softwareVersion
