#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import with_statement
import sys
import serial
import threading
import struct
import time
import unittest
import Queue
import glob
import os.path

from errors import *
from cmds import Command, general

__all__ = "Measurement, Comm"

TIMEOUT = .2  #: TimeOut in s
EOLCHAR = "\r"  #: End of line as OsTech drivers expect
ESCAPE = '\x1B'

def pack(value):
    """Converts a value to a format OsTech drivers understand.
    Numbers are forced to fit into 9 characters.
    """
    if type(value) == bool:
        if value: result = "R"
        else: result = "S"
    elif type(value) in [int, long]:
        result = "%u" % max(0, min(65535, value))
    elif type(value) == float:
        for i in range(9, 0, -1):
            result = "%.*g" % (i, value)
            if len(result) <= 9: break
    else:
        raise TypeError
    return result

class Measurement(object):
    """Measurement object including the measurement value, the command name,
    its parameter and the time of measurement.
    """
    def __init__(self, commands, parameter, mValue, mTime=None):
        self.command = commands.name
        self.parameter = parameter
        self.value = mValue
        if mTime is None:
            self.time = time.time()  # now
        else:
            self.time = mTime

    def dict(self):
        return {
            "command": self.command,
            "parameter": self.parameter,
            "value": repr(self.value),
            "time": repr(self.time),
        }

class CommErrorStats(object):
    def __init__(self):
        self.clear()

    def clear(self):
        self.numCommandSent = 0
        self.numCommandsTimedOut = 0
        self.numCommandsEchoDiffered = 0
        self.numCommandsChecksumWrong = 0
        self.numCommandsRetried = 0

    def command_was_sent(self):
        self.numCommandSent += 1

    def command_timed_out(self):
        self.numCommandsTimedOut += 1

    def command_echo_differed(self):
        self.numCommandsEchoDiffered += 1

    def command_checksum_wrong(self):
        self.numCommandsChecksumWrong += 1

    def command_retried(self):
        self.numCommandsRetried += 1

    def get_num_commands_sent(self):
        return self.numCommandSent

    def get_num_commands_timed_out(self):
        return self.numCommandsTimedOut

    def get_num_commands_echo_differed(self):
        return self.numCommandsEchoDiffered

    def get_num_commands_checksum_wrong(self):
        return self.numCommandsChecksumWrong

    def get_num_commands_retried(self):
        return self.numCommandsRetried

    def get_num_comm_errors(self):
        return self.get_num_commands_timed_out() + \
            self.get_num_commands_echo_differed() + \
            self.get_num_commands_checksum_wrong()


class Comm(object):
    """Communication with OsTech drivers."""
    usedPorts = []
    usedPortsLock = threading.RLock()

    def __init__(self, port=0, idleThread=False):
        self.port = port

        # init serial port
        self.isConnected = False
        self._ser = serial.Serial(timeout=TIMEOUT, writeTimeout=TIMEOUT)
        self._ser.port = port
        self._serLock = threading.RLock()
        self._stats = CommErrorStats()

    def __del__(self):
        self.disconnect()

    def _flush_or_idle(self, skipEscape=False):
        self._ser.flushInput()
        # mark port as unused
        with self.usedPortsLock:
            if self._ser.port in self.usedPorts:
                self.usedPorts.remove(self._ser.port)

    def log(self, message):
        print message

    def connect(self):
        """Establishes a connection to a driver."""
        # mark port as used
        with self.usedPortsLock:
            if self.port not in self.usedPorts:
                self.usedPorts.append(self.port)
            else:
                raise OsTechPortInUseError(self.port)
        with self._serLock:
            self._ser.open()
            if not self._ser.isOpen():
                return
            self.isConnected = True
            self._ser.flushInput()
        self._stats.clear()

    def detect_driver_once(self):
        result = False
        response = ""
        with self._serLock:
            time.sleep(TIMEOUT)
            self._flush_or_idle()
            self._ser.write(ESCAPE)
            response = self._ser.read(1)
        if response == "":
            return result
        if response.count(ESCAPE) > 0:
            result = True
        if response[0] == ESCAPE:
            response = response[1:]
        return result

    def disconnect(self):
        """Closes connection to driver."""
        with self._serLock:
            self._ser.close()
        self.isConnected = False
        # mark port as unused
        with self.usedPortsLock:
            if self.port in self.usedPorts:
                self.usedPorts.remove(self.port)

    def is_connected(self):
        """Returns if connected with a driver."""
        return self.isConnected


    def sendTextCmd(self, cmdStr, returnEcho=False):
        """Sends command to driver in standard mode and returns answer
        string.

        """
        cmd = Command(name=cmdStr, typ=str)
        with self._serLock:
            self.setBinaryMode(False)
            r = self.sendCmd(cmd, returnEcho=returnEcho)
            self.setBinaryMode(True)
        if returnEcho:
            echo, response = r
        else:
            response = r
        if response.value == "":
            self._stats.command_timed_out()
            raise serial.SerialTimeoutException
        if returnEcho:
            return (echo, response.value)
        return response.value

    def setBinaryMode(self, binary=True):
        """Enables or disables binary mode."""
        with self._serLock:
            if binary:
                self.sendCmd(Command(name="GMS", typ=int), 8)
            else:
                self.sendCmd(Command(name="GMC", typ=str), 8)

    def readLine(self):
        line = ""
        if sys.version_info >= (2, 7):
            eolRead = False
            while (not eolRead):
                c = "" + self._ser.read(1)
                if len(c) == 0:
                    return line
                eolRead = c == EOLCHAR
                line += c
        else:
            line = self._ser.readline(eol=EOLCHAR);
        return line

    def sendCmd(self, cmd, newValue=None, retry=True, returnEcho=False,
        flush=True):
        """Sends command to driver and checks echo."""
        cmdStr = cmd.name
        if newValue is not None:
            cmdStr += pack(newValue)
        cmdStr += EOLCHAR
        echo = ""
        with self._serLock:
            for i in range(2):
                try:
                    self._flush_or_idle()
                    self._ser.write(cmdStr)
                    self._stats.command_was_sent()
                    echo = self.readLine()
                    if echo == "":
                        self._stats.command_timed_out()
                        raise serial.SerialTimeoutException
                    if echo != cmdStr.upper():
                        self.log("<<\"%s\" != >>\"%s\"\n" % (repr(echo), repr(cmdStr.upper())))
                        sys.stdout.flush()
                        self._stats.command_echo_differed()
                        raise OsTechEchoDiffersError(cmd.name)
                    if cmd.typ is not None:
                        value = self._readValue(cmd.typ)
                    else:
                        value = None
                    if flush:
                        self._flush_or_idle()
                except (serial.SerialException, OsTechError), e:
                    if not retry:
                        raise
                    else:
                        self._stats.command_retried()
                        self.log("KOMMUNIKATIONSFEHLER %d %s %.3f" % (i + 1, cmdStr[:-1] + str(e), time.time()))
                        time.sleep(.1)
                        self._ser.write(chr(27))
                        time.sleep(.1)
                        if i > 0:
                            raise
                else:
                    break

        result = Measurement(cmd, newValue, value)
        if returnEcho:
            return (echo, result)
        return result

    def setProgrammingMode(self):
        """Put the driver in programming mode."""
        self._ser.flushInput()
        self._ser.write("GPROG\r")
        self.disconnect()

    def readBuildId(self):
        cmd = Command("DB", str, False, False)
        with self._serLock:
            self.setBinaryMode(False)
            response = self.sendCmd(cmd, flush=False)
            response2 = self.readLine()
            self.setBinaryMode(True)
        if response.value == "":
            self._stats.command_timed_out()
            raise serial.SerialTimeoutException
        response = response.value + response2
        response = response.split(EOLCHAR)[:2]
        print repr(response)
        return response

    def _readValue(self, readType):
        """Reads a value of the given type in OsTech driver representation from
        the serial connection. Then the value is converted accordingly and
        returned. If no type is given a string is read until EOLCHAR.
        """
        with self._serLock:
            if readType == bool:
                response = self._ser.read(1)
                if len(response) != 1:
                    self._stats.command_timed_out()
                    raise serial.SerialTimeoutException
                if response == "\xAA":
                    value = True
                elif response == "\x55":
                    value = False
                else:
                    self._stats.command_checksum_wrong()
                    raise OsTechChecksumWrongError
            elif readType == float:
                response = self._ser.read(5)
                if len(response) != 5:
                    self._stats.command_timed_out()
                    raise serial.SerialTimeoutException
                value = struct.unpack('>fx', response)[0]
                chkRead = ord(response[4])
                chkCalc = (0x55 + sum(map(ord, response[:4]))) % 256
                if chkRead != chkCalc:
                    self._stats.command_checksum_wrong()
                    raise OsTechChecksumWrongError
            elif readType == int:
                response = self._ser.read(3)
                if len(response) != 3:
                    self._stats.command_timed_out()
                    raise serial.SerialTimeoutException
                value = struct.unpack('>Hx', response)[0]
                chkRead = ord(response[2])
                chkCalc = (0x55 + sum(map(ord, response[:2]))) % 256
                if chkRead != chkCalc:
                    self._stats.command_checksum_wrong()
                    raise OsTechChecksumWrongError
            else:
                value = self.readLine()
                if value == "":
                    self._stats.command_timed_out()
                    raise serial.SerialTimeoutException
        return value
### end class driver


def readlink_abs(link):
    assert (os.path.islink(link))
    path = os.readlink(link)
    if os.path.isabs(path):
        return path
    return os.path.abspath(os.path.join(os.path.dirname(link), path))


def enumerate_serial_ports():
    ports = []
    if sys.platform.startswith("linux"):
        links = glob.glob("/dev/serial/by-id/*")
        ports = [readlink_abs(link) for link in links if os.path.islink(link)]
    elif sys.platform.startswith("win"):
        if sys.version_info.major > 2:
            import winreg
        else:
            import _winreg as winreg
        path = 'HARDWARE\\DEVICEMAP\\SERIALCOMM'
        try:
            key = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, path)
        except WindowsError:
            return []
        numPorts = winreg.QueryInfoKey(key)[1]
        ports = [bytes(winreg.EnumValue(key, i)[1]) for i in range(numPorts)]
    ports.sort()
    return ports

if __name__ == "__main__":
    print enumerate_serial_ports()

