#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Command class and data structure of all commands drivers support"""


class Command(object):
    """Command object containing a driver command and information about it.
    Attributes:
        name: The command's name
        typ: data type of the command's return value or parameter (bool, int,
            float)
        writable: True means value can be changed, i.e. a parameter is accepted
        state: True means that the command posesses a state and doesn't merely
            carry out an action.
    """

    def __init__(self, name, typ, writeable=True, state=True):
        self.name = name
        self.typ = typ
        self.writeable = writeable
        self.state = state


class CommandDict(dict):
    def __init__(self, mapping={}, *seq, **kwargs):
        if isinstance(mapping, dict):
            for key, value in mapping.items():
                self.__setitem__(None, value)
        elif isinstance(mapping, list):
            for value in mapping:
                self.__setitem__(None, value)
        for key, value in seq:
            self.__setitem__(None, value)
        for key, value in kwargs.items():
            self.__setitem__(None, value)

    def __contains__(self, key):
        return super(CommandDict, self).__contains__(key.upper())

    def __delitem__(self, key):
        return super(CommandDict, self).__delitem__(key.upper())

    def __getitem__(self, key):
        return super(CommandDict, self).__getitem__(key.upper())

    def __setitem__(self, key, value):
        if isinstance(value, Command):
            return super(CommandDict, self).__setitem__(
                value.name.upper(), value)
        else:
            raise TypeError

    def append(self, value):
        return self.__setitem__(None, value)

    def extend(self, iterable):
        for value in iterable:
            self.append(value)


general = CommandDict([
    Command(
        name="GVS",
        typ=int,
        writeable=False,
        state=True
        ),
    Command(
        name="GVT",
        typ=int,
        writeable=False,
        state=True
        ),
    Command(
        name="GVN",
        typ=int,
        writeable=False,
        state=True
        ),
    Command(
        name="GM",
        typ=int,
        writeable=False,
        state=True
        ),
    Command(
        name="GMC",
        typ=int,
        writeable=True,
        state=False
        ),
    Command(
        name="GMS",
        typ=int,
        writeable=True,
        state=False
        ),
    ])


all = CommandDict()
all.extend(general.values())

if __name__ == "__main__":
    print "All commands: " + str(all)
