#include <SerialStream.h>
#include <iostream>
#include <unistd.h>
#include <cstdlib>
#include <cstring>
#include "serial.h"


Serial::Serial(const char * const port) {
    using namespace LibSerial;
    stream.Open(port);
    if (!stream.good())
    {
        std::cerr << "[" << __FILE__ << ":" << __LINE__ << "] "
                  << "Error: Could not open serial port." 
                  << std::endl;
        exit(1);
    }
    // Set the baud rate of the serial port.
    stream.SetBaudRate(SerialStreamBuf::BAUD_9600);
    if (!stream.good())
    {
        std::cerr << "Error: Could not set the baud rate." << std::endl;
        exit(1);
    }
    // Set the number of data bits.
    stream.SetCharSize(SerialStreamBuf::CHAR_SIZE_8);
    if (!stream.good())
    {
        std::cerr << "Error: Could not set the character size." << std::endl;
        exit(1);
    }
    // Disable parity.
    stream.SetParity(SerialStreamBuf::PARITY_NONE);
    if (!stream.good())
    {
        std::cerr << "Error: Could not disable the parity." << std::endl;
        exit(1);
    }
    // Set the number of stop bits.
    stream.SetNumOfStopBits(1);
    if (!stream.good())
    {
        std::cerr << "Error: Could not set the number of stop bits."
                  << std::endl;
        exit(1);
    }
    // Turn off hardware flow control.
    stream.SetFlowControl(SerialStreamBuf::FLOW_CONTROL_NONE);
    if (!stream.good())
    {
        std::cerr << "Error: Could not use hardware flow control."
                  << std::endl;
        exit(1);
    }
    // Do not skip whitespace characters while reading from the
    // serial port.
    stream.unsetf(std::ios_base::skipws);
}


/// Liest Daten vom seriellen Port
/**
 * Liest Daten und schreibt sie in Puffer buf. Maximal length - 1 Bytes werden
 * gelesen. Gelesene Daten werden durch \0 abgeschlossen. Nach endChar wird
 * abgebrochen, endChar ist in Ausgabe enthalten. Nach msTimeout ms wird
 * abgebrochen.
 * @param buf Puffer, in den gelesene Daten geschrieben werden.
 * @param length Laenge des Puffers.
 * @param msTimeout Timeout in ms
 * @param endChar Lesen stoppt, nachdem dieses Zeichen gelesen wurde.
 */
void Serial::read(char *buf, u8 length, u16 msTimeout, const char *endChar)
{
    if (NULL == buf || 0 == length) return;
    u32 usWaited = 0;
    u16 usWaitTime = 100;
    bool done = false;
    u8 numRead = 0;
    buf[0] = 0;

    while (!done) {
        if (0 == stream.rdbuf()->in_avail()) {
            usWaited += usWaitTime;
            usleep(usWaitTime);
        }
        if (stream.rdbuf()->in_avail() > 0) {
            char next_byte;
            stream.get(next_byte);
            buf[numRead++] = next_byte;
            if (NULL != endChar && *endChar == next_byte) done = true;
        }
        if (done || numRead >= length - 1 || usWaited > 1000 * msTimeout) {
            buf[numRead] = 0;
            done = true;
        }
    }
}


void Serial::writeLine(const char *text)
{
    if (NULL == text) return;
    if (0 == strlen(text)) return;
    stream.write(text, strlen(text));
    if ('\r' != text[strlen(text) - 1]) {
        stream.write("\r", 1);
    }
}


void Serial::flush(u16 msTimeout)
{
    usleep(1000 * msTimeout);
    while (0 < stream.rdbuf()->in_avail()) {
        char ch;
        stream.get(ch);
    }
}
