#include "driver.h"
#include <cstring>
#include <cctype>
#include <cstdio>

Driver::Driver(Serial *serial) : serial(serial)
{
    serial->flush(msFlush);
    // Binärmodus anschalten
    serial->writeLine("\x1B" "GMS8\r");
    flush();
    serial->writeLine("\x1B" "GMS8\r");
    flush();
}


void Driver::flush()
{
    serial->flush(msFlush);
}


bool Driver::sendOnlyCmd(const char *cmd, const char *prefix)
{
    if (NULL == cmd) return false;
    u8 prefixLength = 0;
    if (NULL != prefix) prefixLength = std::strlen(prefix);
    const u8 bufLength = prefixLength + std::strlen(cmd) + 2;
    char writeBuf[bufLength];
    writeBuf[0] = '\0';
    char readBuf[bufLength];
    if (NULL != prefix) {
        std::memcpy(writeBuf, prefix, prefixLength + 1);
    }
    std::strncat(writeBuf, cmd, std::strlen(cmd) + 1);
    // ggf. Linebreak anhängen
    const char *linebreak = "\r";
    char *endOfLine = std::strchr(writeBuf, *linebreak);
    if (NULL == endOfLine) {
        std::strncat(writeBuf, linebreak, 2);
    } else {
        // Befehl endet nach Linebreak
        endOfLine[1] = '\0';
    }
    bool again = false;
    do {
        serial->writeLine(writeBuf);
        serial->read(readBuf, bufLength, msTimeout, linebreak);
        for (u8 i = 0; i < std::strlen(cmd); i++) {
            if (std::toupper(writeBuf[i] != readBuf[i])) {
                serial->writeLine("\x1B");
                flush();
                if (again) return false;
                again = true;
                break;
            }
        }
    } while (again);
    return true;
}


bool Driver::sendCmd(const char *cmd, const char *prefix)
{
    bool result = sendOnlyCmd(cmd, prefix);
    flush();
    return result;
}


bool Driver::sendCmd(const char *cmd, const char *prefix, const bool *value, bool *result)
{
    const u8 len = std::strlen(cmd) + 2;
    char buf[len];
    std::memcpy(buf, cmd, std::strlen(cmd) + 1);
    if (NULL != value) {
        if (*value) {
            std::strncat(buf, "R", 1);
        } else {
            std::strncat(buf, "S", 1);
        }
    }
    if (!sendOnlyCmd(buf, prefix)) return false;
    if (NULL != result) {
        return readValue(result);
    }
    flush();
    return true;
}


bool Driver::sendCmd(const char *cmd, const char *prefix, const u16 *value, u16 *result)
{
    const u8 numberLen = 8;
    char number[numberLen];
    const u8 len = std::strlen(cmd) + numberLen + 2;
    char buf[len];
    std::memcpy(buf, cmd, std::strlen(cmd) + 1);
    if (NULL != value) {
        std::snprintf(number, numberLen, "%u", *value);
        std::strncat(buf, number, numberLen);
    }
    if (!sendOnlyCmd(buf, prefix)) return false;
    if (NULL != result) {
        return readValue(result);
    }
    flush();
    return true;
}


bool Driver::sendCmd(const char *cmd, const char *prefix, const float *value, float *result)
{
    const u8 numberLen = 11;
    char number[numberLen];
    const u8 len = std::strlen(cmd) + numberLen + 2;
    char buf[len];
    std::memcpy(buf, cmd, std::strlen(cmd) + 1);
    if (NULL != value) {
        for (u8 i = 9; i > 0; i--) {
            std::snprintf(number, numberLen, "%.*g", i, *value);
            if (std::strlen(number) <= 9) break;
        }
        std::strncat(buf, number, numberLen);
    }
    if (!sendOnlyCmd(buf, prefix)) return false;
    if (NULL != result) {
        return readValue(result);
    }
    flush();
    return true;
}


bool Driver::readValue(bool *value)
{
    const u8 bufLength = 2;
    char buf[bufLength] = {};
    serial->read(buf, bufLength, msTimeout);
    if (0 != buf[1]) return false;
    switch ((u8) buf[0]) {
    case 0x55:
        *value = false;
        return true;
    case 0xAA:
        *value = true;
        return true;
    }
    return false;
}


bool Driver::readValue(u16 *value)
{
    const u8 bufLength = 4;
    char buf[bufLength] = {};
    serial->read(buf, bufLength, msTimeout);
    union {
        u16 u;
        u8 c[2];
    } intBytes;
    u8 checkSum = 0x55;
    for (u8 i = 0; i < 2; i++) {
        // Protokoll: Big Endian, PC und STM32: Little Endian -> vertauschen
        intBytes.c[1 - i] = buf[i];
        checkSum += buf[i];
    }
    if (checkSum == (u16) (buf[2] & 0xFF)) {
        *value = intBytes.u;
        return true;
    } else return false;
}


bool Driver::readValue(float *value)
{
    const u8 bufLength = 6;
    char buf[bufLength] = {};
    serial->read(buf, bufLength, msTimeout);
    union {
        float f;
        u8 c[4];
    } floatBytes;
    u8 checkSum = 0x55;
    for (u8 i = 0; i < 4; i++) {
        // Protokoll: Big Endian, PC und STM32: Little Endian -> vertauschen
        floatBytes.c[3 - i] = buf[i];
        checkSum += buf[i];
    }
    if (checkSum == (u16) (buf[4] & 0xFF)) {
        *value = floatBytes.f;
        return true;
    } else return false;
}
